/* hashtable.h
 *
 * Copyright (C) 2001 Theppitak Karoonboonyanan,
 *   National Electronics and Computer Technology Center
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.  
 */

//
// hashtable.h - associative memory class template
// Created: 27 Apr 1999
// Author:  Theppitak Karoonboonyanan
//

#ifndef MIDA_UTILS_HASHTABLE_H
#define MIDA_UTILS_HASHTABLE_H

#include <mida/utils/hasher.h>
#include <assert.h>

// Note: assumes long is at least 32 bits.
static const int NumPrimes_ = 28;
static const unsigned long Primes_[NumPrimes_] =
{
    53,         97,         193,       389,       769,
    1543,       3079,       6151,      12289,     24593,
    49157,      98317,      196613,    393241,    786433,
    1572869,    3145739,    6291469,   12582917,  25165843,
    50331653,   100663319,  201326611, 402653189, 805306457, 
    1610612741, 3221225473UL, 4294967291UL
};

// find lowest i : Primes_[i] >= n
inline unsigned long NextPrime_(unsigned long n)
{
    int l = 0, h = NumPrimes_;
    while (l < h - 1) {
        int i = (l + h)/2;
        if (Primes_[i] == n) { return n; }
        else if (Primes_[i] < n) { l = i; }
        else { h = i; }

        // check loop invariant
        assert(Primes_[l] <= Primes_[h]);
        assert(n <= Primes_[h]);
    }
    return Primes_[h];
}

template <class Key, class Val>
class HashTable {
public:  // public type definitions
    typedef Key KeyType;
    typedef Val ValueType;

public:  // public member functions
    HashTable()
        : tableSize_(Primes_[0]), nData_(0)
        { pHashCells_ = new THashCell_[tableSize_]; }
    HashTable(int table_size)
        : tableSize_(NextPrime_(table_size)), nData_(0)
        { pHashCells_ = new THashCell_[tableSize_]; }
    ~HashTable()  { delete[] pHashCells_; }

    const ValueType* operator[](const KeyType& k) const
    {
        int index = lookup_(k);
        return (index == -1) ? 0 : &pHashCells_[index].value;
    }

    ValueType* operator[](const KeyType& k)
    {
        int index = lookup_(k);
        return (index == -1) ? 0 : &pHashCells_[index].value;
    }

    bool add(const KeyType& k, const ValueType& v)
    {
        resize(nData_ + 1);
        return addNoResize_(k, v);
    }
    bool remove(const KeyType& k)
    {
        int index = lookup_(k);
        if (index < 0) { return false; }

        pHashCells_[index].flag = THashCell_::STF_DELETED;

        --nData_;

        return true;
    }
    void resize(int size_hint)
    {
        if (size_hint <= tableSize_) { return; }
        int new_size = NextPrime_(size_hint);
        if (new_size > tableSize_) {
            THashCell_* old_table = pHashCells_;
            pHashCells_ = new THashCell_[new_size];
            int old_table_size = tableSize_;
            tableSize_ = new_size;
            nData_ = 0;
            for (int i = 0; i < old_table_size; ++i) {
                if (old_table[i].flag == THashCell_::STF_USED) {
                    addNoResize_(old_table[i].key, old_table[i].value);
                }
            }
            delete[] old_table;
        }
    }

private:  // private constants & typedefs
    struct THashCell_ {
    public:  // public consts
        enum TFlag_ {
            STF_UNUSED,
            STF_DELETED,
            STF_USED
        };

    public: // public data
        TFlag_     flag;
        KeyType    key;
        ValueType  value;

    public: // public c-tor
        THashCell_() : flag(STF_UNUSED) {}
    };

private:  // private functions
    // lookup_()
    // returns the cell for the key k
    //         -1 if not found
    int lookup_(const KeyType& k) const
    {
       int h = KeyHash<KeyType>(k, tableSize_);
       int i = h, d = 1;
       while (
           (pHashCells_[i].flag == THashCell_::STF_USED
            && pHashCells_[i].key != k)
           || pHashCells_[i].flag == THashCell_::STF_DELETED
       ) {
           i = (i + d) % tableSize_;
           if (i == h) { return -1; }
           d += 2;
       }
       return (pHashCells_[i].flag == THashCell_::STF_USED) ? i : -1;
    }
    // findFreeCell_()
    // returns free cell index
    //         -1 if table full
    //         -2 if k already exists
    int findFreeCell_(const KeyType& k) const
    {
       int h = KeyHash<KeyType>(k, tableSize_);
       int i = h, d = 1;
       while (pHashCells_[i].flag == THashCell_::STF_USED) {
           if (pHashCells_[i].key == k) { return -2; }
           i = (i + d) % tableSize_;
           if (i == h) { return -1; }
           d += 2;
       }
       return i;
    }

    bool addNoResize_(const KeyType& k, const ValueType& v)
    {
        int index = findFreeCell_(k);
        if (index < 0) { return false; }

        pHashCells_[index].flag  = THashCell_::STF_USED;
        pHashCells_[index].key   = k;
        pHashCells_[index].value = v;

        ++nData_;

        return true;
    }

private:  // private data
    int         tableSize_;
    int         nData_;
    THashCell_* pHashCells_;
};

#endif  // MIDA_UTILS_HASHTABLE_H

