/* hasher.h
 *
 * Copyright (C) 2001 Theppitak Karoonboonyanan,
 *   National Electronics and Computer Technology Center
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.  
 */

//
// hasher.h - templates for safe hashing
// Created: 26 Apr 1999
// Author:  Theppitak Karoonboonyanan
// Reference:
//   Michael P. Lindner, "Designing for Safe Hashing", C++ Report,
//       November-December 1996.
//

#ifndef MIDA_UTILS_HASHER_H
#define MIDA_UTILS_HASHER_H

typedef unsigned long THash;

class HashStream {
public:  // public functions
    HashStream(THash modulo) : h_(0), modulo_(modulo) {}

    HashStream& operator<<(char c)
    {
        appendHashKey_((unsigned char&)c);  return *this;
    }
    HashStream& operator<<(unsigned char c)
    {
        appendHashKey_(c);  return *this;
    }
    HashStream& operator<<(int i)
    {
        for (int b = sizeof(int); b > 0; --b) {
            appendHashKey_((unsigned char)(i & 0xff));
            i >>= 8;
        }
        return *this;
    }
    HashStream& operator<<(unsigned int i)
    {
        for (int b = sizeof(unsigned int); b > 0; --b) {
            appendHashKey_((unsigned char)(i & 0xff));
            i >>= 8;
        }
        return *this;
    }
    HashStream& operator<<(short i)
    {
        for (int b = sizeof(short); b > 0; --b) {
            appendHashKey_((unsigned char)(i & 0xff));
            i >>= 8;
        }
        return *this;
    }
    HashStream& operator<<(unsigned short i)
    {
        for (int b = sizeof(unsigned short); b > 0; --b) {
            appendHashKey_((unsigned char)(i & 0xff));
            i >>= 8;
        }
        return *this;
    }
    HashStream& operator<<(long l)
    {
        for (int b = sizeof(long); b > 0; --b) {
            appendHashKey_((unsigned char)(l & 0xff));
            l >>= 8;
        }
        return *this;
    }
    HashStream& operator<<(unsigned long l)
    {
        for (int b = sizeof(unsigned long); b > 0; --b) {
            appendHashKey_((unsigned char)(l & 0xff));
            l >>= 8;
        }
        return *this;
    }
    HashStream& operator<<(const char* s)
    {
        for ( ; *s; ++s) {
            appendHashKey_(*(const unsigned char*)s);
        }
        return *this;
    }
    HashStream& operator<<(const unsigned char* s)
    {
        for ( ; *s; ++s) {
            appendHashKey_(*s);
        }
        return *this;
    }

    THash hashVal() const { return h_; }

private:  // private functions
    void appendHashKey_(unsigned char c) { h_ = (h_*3 + c) % modulo_; }

private:  // private data
    THash h_;
    THash modulo_;
};


template <class Key>
inline THash KeyHash(const Key& k, THash modulo)
{
    HashStream h(modulo);
    h << k;
    return h.hashVal();
}


#endif  // MIDA_UTILS_HASHER_H

