#include "gswitchit.h"

#include <gdk/gdkx.h>

#include <stdio.h>
#include <limits.h>

#include "X11/keysym.h"

#include "switchcuts.h"

#include "../xklavier/xklavier.h"

#define SWITCHCUT_DEF( name, keys, mods )   { name, keys, mods },

Switchcut switchcuts[] =
{
  #include "switchcuts.inc"
};

int total_switchcuts = sizeof ( switchcuts )/sizeof ( switchcuts[0] );

void GSwitchItConfigFreeImages( GSwitchItConfig* sic )
{
  int i;
  for ( i = XkbNumKbdGroups;
        --i>=0; )
  {
    if ( sic->images[i] )
    {
      gdk_pixbuf_unref ( sic->images[i] );
      sic->images[i] = NULL;
    }
  }

  if ( sic->activeImage )
  {
    gdk_pixmap_unref ( sic->activeImage );
    sic->activeImage = NULL;
  }
  if ( sic->activeMask )
  {
    gdk_bitmap_unref ( sic->activeMask );
    sic->activeMask = NULL;
  }
}

void GSwitchItConfigLoadImages( GSwitchItConfig* sic )
{
  int i;

  for ( i = XkbNumKbdGroups;
        --i>=0; )
  {
    if ( sic->imageFiles[i] != NULL )
      sic->images[i] = gdk_pixbuf_new_from_file( sic->imageFiles[i] );
  }
}

void GSwitchItConfigInit( GSwitchItConfig* sic, gboolean doLoad )
{
  GError * gerror = NULL;

  sic->confClient = gconf_client_get_default();

  gconf_client_add_dir( sic->confClient,
            "/apps/" PACKAGE "/General",
                        GCONF_CLIENT_PRELOAD_NONE,
      &gerror );
  if ( gerror != NULL )
  {
    XklDebug( "err1:%s\n", gerror->message );
    g_error_free( gerror );
    gerror = NULL;
  }
  gconf_client_add_dir( sic->confClient,
            "/apps/" PACKAGE "/Images",
                        GCONF_CLIENT_PRELOAD_NONE,
      &gerror );
  if ( gerror != NULL )
  {
    XklDebug( "err2:%s\n", gerror->message );
    g_error_free( gerror );
    gerror = NULL;
  }

  if ( doLoad )
    GSwitchItConfigLoad( sic );
}

void GSwitchItConfigTerm( GSwitchItConfig* sic, gboolean doSave )
{
  int i;

  if ( doSave )
    GSwitchItConfigSave( sic );

  GSwitchItConfigFreeImages( sic );

  for ( i = XkbNumKbdGroups;
        --i>=0; )
    if ( sic->imageFiles[i] )
    {
      g_free( sic->imageFiles[i] );
      sic->imageFiles[i] = NULL;
    }
  gtk_object_unref( GTK_OBJECT( sic->confClient ) );
}

static char* defaultFiles[XkbNumKbdGroups]=
{
  "us.xpm",
  "th.xpm",
  "us.xpm",
  "us.xpm",
};

void GSwitchItConfigLoad( GSwitchItConfig* sic )
{
  int i;
  GError * gerror = NULL;

  for ( i = XkbNumKbdGroups;
        --i>=0; )
  {
    gchar sz[ ( sizeof CONFIG_KEY_IMAGES_FMT ) + 3 ], *p;

    if ( sic->imageFiles[i] )
      g_free( sic->imageFiles[i] );
    g_snprintf( sz, sizeof(sz), CONFIG_KEY_IMAGES_FMT, i );
    p = gconf_client_get_string( sic->confClient, sz, &gerror );
    if ( p == NULL || gerror != NULL )
    {
      if ( gerror != NULL )
      {
        g_warning( "Error reading configuration:%s\n", gerror->message );
        g_error_free( gerror );
        gerror = NULL;
      }
      // just take default value
      p = gnome_pixmap_file( defaultFiles[i] );
    }

    sic->imageFiles[i] = p;
  }

  sic->secondaries = gconf_client_get_int( sic->confClient,
                                           CONFIG_KEY_SECONDARIES, &gerror );
  if ( gerror != NULL )
  {
    g_warning( "Error reading configuration:%s\n", gerror->message );
    sic->secondaries = 0;
    g_error_free(gerror);
    gerror = NULL;
  }

  sic->doBeep = gconf_client_get_bool( sic->confClient,
                                       CONFIG_KEY_BEEP, &gerror );
  if ( gerror != NULL )
  {
    g_warning( "Error reading configuration:%s\n", gerror->message );
    sic->doBeep = FALSE;
    g_error_free(gerror);
    gerror = NULL;
  }

  sic->layoutPerApp = gconf_client_get_bool( sic->confClient,
                                       CONFIG_KEY_GROUP_PER_WINDOW, &gerror );
  if ( gerror != NULL )
  {
    g_warning( "Error reading configuration:%s\n", gerror->message );
    sic->layoutPerApp = FALSE;
    g_error_free(gerror);
    gerror = NULL;
  }

  sic->switchcutId = gconf_client_get_int( sic->confClient,
                                            CONFIG_KEY_SWITCHCUT_ID, &gerror );
  if ( gerror != NULL )
  {
    g_warning( "Error reading configuration:%s\n", gerror->message );
    sic->switchcutId = 0;
    g_error_free(gerror);
    gerror = NULL;
  }

  sic->switchcutKeysym = gconf_client_get_int( sic->confClient,
                                            CONFIG_KEY_SWITCHCUT_KEYSYM, &gerror );
  if ( gerror != NULL )
  {
    g_warning( "Error reading configuration:%s\n", gerror->message );
    if( sic->switchcutId == total_switchcuts - 1 )
    {
      sic->switchcutId = 0;
    }
    g_error_free(gerror);
    gerror = NULL;
  } else
    sic->switchcutAutoKeycode = XKeysymToKeycode( GDK_DISPLAY(), sic->switchcutKeysym );

  sic->switchcutState = gconf_client_get_int( sic->confClient,
                                            CONFIG_KEY_SWITCHCUT_STATE, &gerror );
  if ( gerror != NULL )
  {
    g_warning( "Error reading configuration:%s\n", gerror->message );
    if( sic->switchcutId == total_switchcuts - 1 )
    {
      sic->switchcutId = 0;
    }
    g_error_free(gerror);
    gerror = NULL;
  }

  if ( sic->switchcutId < 0 ||
       sic->switchcutId >= total_switchcuts )
    sic->switchcutId = 0;

  GSwitchItConfigFreeImages( sic );
  GSwitchItConfigLoadImages( sic );
}

void GSwitchItConfigSave( GSwitchItConfig* sic )
{
  int i;
  GConfChangeSet* cs;
  GError * gerror = NULL;
  cs = gconf_change_set_new();

  for ( i = XkbNumKbdGroups;
        --i>=0; )
  {
    if ( sic->imageFiles[i] != NULL )
    {
      gchar buf[ ( sizeof CONFIG_KEY_IMAGES_FMT ) + 3 ];

      g_snprintf( buf, sizeof(buf), CONFIG_KEY_IMAGES_FMT, i );
      gconf_change_set_set_string( cs, buf, sic->imageFiles[i] );
    }
  }

  gconf_change_set_set_int( cs, CONFIG_KEY_SECONDARIES, sic->secondaries );
  gconf_change_set_set_bool( cs, CONFIG_KEY_BEEP, sic->doBeep );
  gconf_change_set_set_bool( cs, CONFIG_KEY_GROUP_PER_WINDOW, sic->layoutPerApp );
  gconf_change_set_set_int( cs, CONFIG_KEY_SWITCHCUT_KEYSYM, sic->switchcutKeysym );
  gconf_change_set_set_int( cs, CONFIG_KEY_SWITCHCUT_STATE, sic->switchcutState );
  gconf_change_set_set_int( cs, CONFIG_KEY_SWITCHCUT_ID, sic->switchcutId );

  gconf_client_commit_change_set( sic->confClient, cs, TRUE, &gerror );
  if ( gerror != NULL )
  {
    g_warning( "Error saving configuration: %s\n", gerror->message );
    g_error_free( gerror );
    gerror = NULL;
  }
  gconf_change_set_unref( cs );
}

gboolean GSwitchItConfigGrabSwitchcut( GSwitchItConfig* config )
{
  int scid;
  Switchcut * sc;

  scid = config->switchcutId;
  if ( scid <= 0 || scid >= total_switchcuts )
    return FALSE;

  if(scid != total_switchcuts - 1)
  {
    sc = switchcuts + scid;

    if ( !sc->key ) return FALSE;
    return TRUE == XklGrabKey( sc->key, sc->modifiers );
  } else
  {
    return TRUE == XklGrabKey( config->switchcutKeysym, config->switchcutState );
  }
}

gboolean GSwitchItConfigUngrabSwitchcut( GSwitchItConfig* config )
{
  int scid;
  Switchcut * sc;

  scid = config->switchcutId;
  if ( scid <= 0 || scid >= total_switchcuts )
    return FALSE;

  sc = switchcuts + scid;

  return  TRUE == XklUngrabKey( sc->key, sc->modifiers );
}

void GSwitchItConfigLockNextGroup( GSwitchItConfig* config )
{
  int group = XklGetNextGroup();
  GSwitchItConfigApproveSwitch( config, group );
  XklLockGroup( group, False );
}

void GSwitchItConfigLockPrevGroup( GSwitchItConfig* config )
{
  int group = XklGetPrevGroup();
  GSwitchItConfigApproveSwitch( config, group );
  XklLockGroup( group, False );
}

void GSwitchItConfigRestoreGroup( GSwitchItConfig* config )
{
  int group = XklGetRestoreGroup() ;
  GSwitchItConfigApproveSwitch( config, group );
  XklLockGroup( group, False );
}

void GSwitchItConfigApproveSwitch( GSwitchItConfig* config, int group )
{
  XklDebug( "Next switch to group %d will be approved\n" );
  config->switchApproved |= ( 1 << group );
}

gboolean GSwitchItConfigIsSwitchApproved( GSwitchItConfig* config, int group )
{
  int scid;
  int mask = 1 << group;

  if ( config->switchApproved & mask )
  {
     XklDebug( "Approval: %X, will be cleaned for group %d by mask %X\n", 
       config->switchApproved, group, mask );

    config->switchApproved &= ~mask;
    // reset switch!

    return TRUE;
  }

  scid = config->switchcutId;
  // switchcut 0 means default XKB switch
  if ( scid == 0 )
  {
    return TRUE;
  }

  return FALSE;
}

gboolean GSwitchItConfigProcessSwitch( GSwitchItConfig *config, int group )
{
  // check whether this allowed
  if ( ! GSwitchItConfigIsSwitchApproved( config, group ) )
  {
    XklDebug( "!! Switch is not approved, so we'll restore the group !!\n" );
    config->allowSecondary = TRUE;
    config->forceNoBeep = TRUE;
    GSwitchItConfigRestoreGroup( config );
    return FALSE;
  }
  return TRUE;
}

