#include <stdio.h>
#include <stdarg.h>
#include <time.h>

#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xutil.h>
#include <X11/XKBlib.h>
#include <X11/Xlibint.h>

#include "xklavier.h"
#include "xklavier_private.h"

static Display *dpy;
static int scr;

static Window root;

static Atom WM_NAME, WM_STATE, XKLAVIER_STATE;

static XklState curState;

static Window curClient;

static WinCallback winCallback = NULL;
static void *winCallbackData;

static StateCallback stateCallback = NULL;
static void *stateCallbackData;

static unsigned numGroups;

static char* groupNames[ XkbNumKbdGroups ];

static int xkbEventType, xkbError;

static const char * lastError;

static Bool restoreState = False;

static XErrorHandler DefaultErrHandler;

static Bool layoutPerApp = True;

char* XklGetDebugWindowTitle( Window win )
{
  static char sname[ 33 ];
  char *name;
  strcpy( sname, "NULL" );
  if ( win != ( Window ) NULL )
  {
    name = XklGetWindowTitle( win );
    if ( name != NULL )
    {
      snprintf( sname, sizeof( sname ),
                "%.32s", name );
      XFree( name );
    }
  }
  return sname;
}

Window XklGetCurrentWindow()
{
  return curClient;
}

const char ** XklGetGroupNames( void )
{
  return ( const char ** ) groupNames;
}

const char * XklGetLastError()
{
  return lastError;
}

int XklRegisterWindowCallback( WinCallback fun, void *data )
{
  winCallback = fun;
  winCallbackData = data;
  return 0;
}

int XklRegisterStateCallback( StateCallback fun, void *data )
{
  stateCallback = fun;
  stateCallbackData = data;
  return 0;
}

int XklInit( Display * a_dpy )
{
  int opcode;

  DefaultErrHandler = XSetErrorHandler( ( XErrorHandler ) XklErrHandler );

  /* Lets begin */
  XkbQueryExtension( dpy = a_dpy,
                     &opcode, &xkbEventType, &xkbError, NULL, NULL );

  scr = DefaultScreen( dpy );
  root = RootWindow( dpy, scr );
  XklDebug( "xkbEvenType: %X, xkbError: %X, display: %p, root: %ld\n",
            xkbEventType, xkbError, dpy, root );

  XKLAVIER_STATE = XInternAtom( dpy, "XKLAVIER_STATE", False );
  WM_STATE = XInternAtom( dpy, "WM_STATE", False );
  WM_NAME = XInternAtom( dpy, "WM_NAME", False );

  XklLoadInfo();

  return 0;
}

int XklStartListen()
{
  int evtmask;

  evtmask = XkbGroupLockMask |
            XkbModifierStateMask |
            XkbModifierBaseMask |
            XkbModifierLatchMask |
            XkbModifierLockMask;

  //  XGrabServer( dpy );
  /* What events we want */
  XkbSelectEventDetails( dpy,
                         XkbUseCoreKbd,
                         XkbStateNotify,
                         //    evtmask,
                         //    evtmask,
                         XkbAllStateComponentsMask,
                         XkbGroupStateMask );

  XklSelectInput( root, SubstructureNotifyMask );

  XklLoadWindowTree();

  //  XUngrabServer( dpy );
  XFlush( dpy );
  return 0;
}

int XklTerm()
{
  XSetErrorHandler( ( XErrorHandler ) DefaultErrHandler );
  return 0;
}

int XklLoadWindowTree()
{
  Window focused;
  int revert;

  XklReset();

  XklLoadSubtree( root, 0 );

  XGetInputFocus( dpy, &focused, &revert );
  if ( XklGetState( focused, &curState ) )
  {
    XklGetAppWindow( focused, &curClient );
  }
  XklDebug( "initial curClient: %ld, '%s'\n", curClient, XklGetDebugWindowTitle( curClient ) );
  return 0;
}

void XklLoadSubtree( Window window, int level )
{
  Window rwin = ( Window ) NULL,
                parent = ( Window ) NULL,
                         *children = NULL,
                                     *child;
  int num = 0;

  XQueryTree( dpy, window, &rwin, &parent, &children, &num );

  child = children;
  while ( num )
  {
    XklDebug( "Looking at child %ld '%s'\n", *child, XklGetDebugWindowTitle( *child ) );
    if ( XklHasWmState( *child ) )
    {
      XklDebug( "It has WM_STATE so we'll add it\n" );
      XklAddAppWindow( *child, window, True, &curState );
    }
    else
    {
      XklDebug( "It does not have have WM_STATE so we'll not add it\n" );

      if ( level == 0 )
      {
        XklDebug( "But we are at level 0 so we'll spy on it\n" );
        XklSelectInput( *child, FocusChangeMask | PropertyChangeMask );
      }
      else
        XklDebug( "And we are at level %d so we'll not spy on it\n", level );

      XklLoadSubtree( *child, level + 1 );
    }

    child++;
    num--;
  }

  if ( children != NULL )
    XFree( children );
}

int XklFilterEvents( XEvent * xev )
{
  if ( xev->type == xkbEventType )
  {
    XklXkbEvHandler( ( XkbEvent * ) xev );
  }
  else
    switch ( xev->type )
    { /* core events */
    case FocusIn:
      XklFocusInEvHandler( &xev->xfocus );
      break;
    case FocusOut:
      XklFocusOutEvHandler( &xev->xfocus );
      break;
    case PropertyNotify:
      XklPropertyEvHandler( &xev->xproperty );
      break;
    case CreateNotify:
      XklCreateEvHandler( &xev->xcreatewindow );
      break;
    case DestroyNotify:
      XklDebug( "Window %ld destroyed\n", xev->xdestroywindow.window );
      break;
    case UnmapNotify:
      //      XklDebug( "UnmapNotify\n" );
      break;
    case MapNotify:
      //      XklDebug( "MapNotify\n" );
      break;
    case MappingNotify:
      //      XklDebug( "MappingNotify\n" );
      break;
    case GravityNotify:
      //      XklDebug( "GravityNotify\n" );
      break;
    case ReparentNotify:
      //      XklDebug( "ReparentNotify\n" );
      break;  /* Ignore these events */
    default:
      //      XklDebug( "Unknown event %d\n", xev->type );
      break;
    }
  return 0;
}

Bool XklIsSameApp( Window win1, Window win2 )
{
  Window app1, app2;
  return XklGetAppWindow( win1, &app1 ) &&
         XklGetAppWindow( win2, &app2 ) &&
         app1 == app2;
}

void XklReset()
{
  XkbStateRec state;

  curState.group = 0;
  if ( Success == XkbGetState( dpy, XkbUseCoreKbd, &state ) )
    curState.group = state.locked_group;

  curClient = 0;
}

void XklAddAppWindow( Window appWin, Window parent, Bool force, XklState * init_state )
{
  XklState state = *init_state;
  Bool have_state;

  if ( appWin == root )
    XklDebug( "??? root app win ??\n" );

  XklDebug( "Trying to add window %ld/%s with group %d\n",
            appWin,
            XklGetDebugWindowTitle( appWin ),
            init_state->group );

  have_state = XklGetAppState( appWin, &state );

  if ( have_state && !force )
  {
    XklDebug( "The window %ld does not require to be added, it already has the xklavier state \n", appWin );
    return ;
  }

  XklSaveAppState( appWin, &state );
  XklSelectInput( appWin, FocusChangeMask | PropertyChangeMask );

  if ( parent == ( Window ) NULL )
    parent = XklGetRegisteredParent( appWin );

  if ( winCallback != NULL )
    ( *winCallback ) ( appWin, parent, winCallbackData );

  XklDebug( "done\n" );
}

Bool XklGetAppWindow( Window win, Window * core )
{
  Window parent = ( Window ) NULL,
                  rwin = ( Window ) NULL,
                         *children = NULL,
                                     *child;
  int num = 0;
  Status result;
  Bool rv;

  static Bool isFirstLevel = True;

  if ( win == ( Window ) NULL ||
          win == root )
  {
    *core = win;
    lastError = "The window is either 0 or root";
    return False;
  }

  if ( XklHasWmState( win ) )
  {
    *core = win;
    return True;
  }

  result = XQueryTree( dpy, win, &rwin, &parent, &children, &num );

  if ( result != Success )
  {
    XklDebug( "Could not get tree info for window %ld: %d\n", win, result );
    *core = ( Window ) NULL;
    lastError = "Could not get the tree info";
    return False;
  }

  if ( isFirstLevel )
  {
    child = children;
    while ( num )
    {
      if ( XklHasWmState( *child ) )
      {
        *core = *child;
        if ( children != NULL )
          XFree( children );
        return True;
      }
      child++;
      num--;
    }
  }
  if ( children != NULL )
    XFree( children );

  isFirstLevel = False;
  rv = XklGetAppWindow( parent, core );
  isFirstLevel = True;
  return rv;
}

Bool XklGetState( Window win, XklState* state )
{
  Window appWin;

  if ( !XklGetAppWindow( win, &appWin ) )
  {
    if ( state != NULL )
      state->group = -1;
    return False;
  }

  return XklGetAppState( appWin, state );
}

void XklDelState( Window win )
{
  Window appWin;

  if ( XklGetAppWindow( win, &appWin ) )
    XklDelAppState( appWin );
}

void XklSaveState( Window win, XklState* state )
{
  Window appWin;

  if ( XklGetAppWindow( win, &appWin ) )
    XklSaveAppState( appWin, state );
}

Bool XklHasWmState( Window win )
{   /* ICCCM 4.1.3.1 */
  Atom type = None;
  int format;
  unsigned long nitems;
  unsigned long after;
  unsigned char* data = NULL; /* Helps in the case of BadWindow error */

  XGetWindowProperty( dpy, win, WM_STATE, 0, 0, False, WM_STATE,
                      &type, &format, &nitems, &after, &data );
  if ( data != NULL )
    XFree( data );  /* To avoid an one-byte memory leak because after successfull return
                     * data array always contains at least one nul byte (NULL-equivalent) */
  return type != None;
}

void XklXkbEvHandler( XkbEvent * kev )
{
  int grp;
  Bool foutFlag = False;

  XklDebug( "Xkb event detected, new group %d\n", kev->state.locked_group );

  switch ( kev->any.xkb_type )
  {
  case XkbStateNotify:
    grp = kev->state.locked_group;

    if ( !foutFlag )
    {
      Window focused;
      Window focusedApp;
      int revert;

      XGetInputFocus( dpy, &focused, &revert );
      if ( ( focused == None ) || ( focused == PointerRoot ) )
      {
        XklDebug( "Something with focus: %ld\n", focused );
        break;
      }

      if ( !XklGetAppWindow( focused, &focusedApp ) )
        focusedApp = curClient; //what else can I do

      if ( focusedApp != curClient )
      {
        XklDebug( "Focused window: %ld, '%s'\n", focusedApp,
                  XklGetDebugWindowTitle( focusedApp ) );
        XklDebug( "CurClient: %ld, '%s'\n", curClient,
                  XklGetDebugWindowTitle( curClient ) );
        curState.group = grp;
        XklAddAppWindow( focusedApp, ( Window ) NULL, False, &curState );
        curClient = focusedApp;
      }
    }
    foutFlag = False;

    curState.group = grp;

    if ( stateCallback != NULL )
      ( *stateCallback ) ( grp, restoreState, stateCallbackData );
    restoreState = False;

    XklSaveState( curClient, &curState );

    break;
  default:
    break;
  }
}

void XklFocusInEvHandler( XFocusChangeEvent* fev )
{
  Window win;
  Window winApp;

  win = fev->window;

  XklDebug( "Window %ld, '%s' has got the focus\n", win,
            XklGetDebugWindowTitle( win ) );

  if ( !XklGetAppWindow( win, &winApp ) )
  {
    XklDebug( "Strange, but could not get the app window\n" );
    return ;
  }

  XklDebug( "Appwin %ld, '%s' has got the focus\n", winApp,
            XklGetDebugWindowTitle( winApp ) );

  if ( XklGetState( winApp, &curState ) )
  {
    curClient = winApp;
    if ( layoutPerApp )
    {
      XklDebug( "Restoring the group %d after gaining the focus\n", curState.group );
      XklLockGroup( curState.group, True );
    } else
      XklDebug( "Not restoring the group %d after gaining the focus: global layout is active\n", curState.group );
    
  }
  else
  {
    XklDebug( "But it does not have xklavier_state\n" );
    if ( XklHasWmState( win ) )
    {
      XklDebug( "But it does have wm_state so we'll add it\n" );
      curClient = winApp;
      XklAddAppWindow( curClient, ( Window ) NULL, False, &curState );
    }
    else
      XklDebug( "And it does have wm_state either\n" );
  }
}

void XklFocusOutEvHandler( XFocusChangeEvent* fev )
{
  XklDebug( "Window %ld, '%s' has lost the focus\n", fev->window, XklGetDebugWindowTitle( fev->window ) );
}

void XklPropertyEvHandler( XPropertyEvent* pev )
{
  Bool hasXklState;
  char * atomName;

  atomName = XGetAtomName( dpy, pev->atom );
  if ( atomName != NULL )
  {
    XklDebug( "The property '%s' changed for %ld\n", atomName, pev->window );
    XFree( atomName );
  }
  else
    XklDebug( "Some magic property changed for %ld\n", pev->window );

  hasXklState = XklGetState( pev->window, NULL );

  if ( pev->atom == WM_STATE )
  {
    if ( pev->state == PropertyNewValue )
    {
      XklDebug( "New value of WM_STATE on window 0x%x\n", pev->window );
      if ( !hasXklState )                  /* Is this event the first or not? */
      {
        XklAddAppWindow( pev->window, ( Window ) NULL, False, &curState );
      }
    }
    else
    { /* ev->xproperty.state == PropertyDelete, either client or WM can remove it, ICCCM 4.1.3.1 */
      XklSelectInput( pev->window, PropertyChangeMask );
      if ( hasXklState )
      {
        XklDelState( pev->window );
      }
    }
  }
  return ;
}

void XklErrHandler( Display* dpy, XErrorEvent* evt )
{
  if ( evt->error_code == BadWindow )
  {
    // in most cases this means we are late:)
#if 0
    XklDebug( "ERROR: %p, %ld, %d, %d, %d\n",
              dpy,
              ( unsigned long ) evt->resourceid,
              ( int ) evt->error_code,
              ( int ) evt->request_code,
              ( int ) evt->minor_code );
#endif

  }
  else
    ( *DefaultErrHandler ) ( dpy, evt );

}

void XklCreateEvHandler( XCreateWindowEvent* cev )
{
  XklDebug( "Under-root window %ld/%s (%d,%d,%d x %d) is created\n",
            cev->window,
            XklGetDebugWindowTitle( cev->window ),
            cev->x, cev->y, cev->width, cev->height );

  if ( !cev->override_redirect )
  { /* ICCCM 4.1.6: override-redirect is NOT private to
                                                          * client (and must not be changed - am I right?) */
    /* We really need only PropertyChangeMask on this window but even in the case of
     * local server we can lose PropertyNotify events (the trip time for this CreateNotify
     * event + SelectInput request is not zero) and we definitely will (my system DO)
     * lose FocusIn/Out events after the following call of PropertyNotifyHandler.
     * So I just decided to purify this extra FocusChangeMask in the FocusIn/OutHandler. */
    XklSelectInput( cev->window, PropertyChangeMask | FocusChangeMask );
  }
}

Window XklGetRegisteredParent( Window win )
{
  Window parent = ( Window ) NULL,
                  rw = ( Window ) NULL,
                       *children = NULL;
  unsigned nchildren = 0;
  Status result;

  result = XQueryTree( dpy, win, &rw, &parent, &children, &nchildren );
  if ( children != NULL )
    XFree( children );
  return result == Success ? parent : ( Window ) NULL;
}

void XklLoadInfo()
{
  XkbDescPtr kbinfo;
  Status status;

  kbinfo = XkbGetMap( dpy,
                      0,  //XkbAllComponentsMask,
                      XkbUseCoreKbd );
  if ( kbinfo != NULL )
  {
    status = XkbGetControls( dpy, XkbSlowKeysMask, kbinfo );

    if ( kbinfo->ctrls != NULL && status == Success )
    {
      numGroups = kbinfo->ctrls->num_groups;
      if ( numGroups > XkbNumKbdGroups )
        numGroups = XkbNumKbdGroups;
    }
    status = XkbGetNames( dpy, XkbGroupNamesMask, kbinfo );
    if ( kbinfo->names != NULL && status == Success )
    {
      int i;
      for ( i = 0; i < numGroups; i++ )
      {
        //!! never freed
        if ( kbinfo->names->groups != 0L )
          groupNames[ i ] = XGetAtomName( dpy,
                                          kbinfo->names->groups[ i ] );
        else
          groupNames[ i ] = "-";
        XklDebug ( "group %d has name %s\n", i, groupNames[ i ] );
      }
    }

    XkbGetControls( dpy, XkbAllControlsMask, kbinfo );

    XkbFreeKeyboard( kbinfo, XkbAllControlsMask, True );
  }
}

unsigned XklGetNumGroups()
{
  return numGroups;
}

int XklGetNextGroup()
{
  return ( curState.group + 1 ) % numGroups;
}

int XklGetPrevGroup()
{
  return ( curState.group + numGroups - 1 ) % numGroups;
}

int XklGetRestoreGroup()
{
  XklState state;
  if ( curClient == ( Window ) NULL )
  {
    XklDebug( "cannot restore without current client\n" );
  }
  else
    if ( XklGetState( curClient, &state ) )
    {
      return state.group;
    }
    else
      XklDebug ( "Unbelievable: current client %ld, '%s' has no group\n", curClient, XklGetDebugWindowTitle( curClient ) );
  return 0;
}

Bool XklGrabKey( int key, unsigned modifiers )
{
  int keyCode;
  int retCode;
  char * keyName;

  keyCode = XKeysymToKeycode( dpy, key );
  keyName = XKeysymToString( key );

  XklDebug( "listen to the key %X(%d/%s)/%d\n", key, keyCode, keyName, modifiers );

  if ( ( KeyCode ) NULL == keyCode )
    return False;

  retCode = XGrabKey( dpy, keyCode, modifiers, root,
                      True, GrabModeAsync, GrabModeAsync );

  XklDebug( "trying to listen: %d\n", retCode );

  return ( Success == retCode );
}

Bool XklUngrabKey( int key, unsigned modifiers )
{
  int keyCode;

  keyCode = XKeysymToKeycode( dpy, key );

  if ( ( KeyCode ) NULL == keyCode )
    return False;

  return Success == XUngrabKey( dpy, keyCode, 0, root );
}

void _XklDebug( const char file[], const char function[], const char format[], ... )
{
  time_t now;
  va_list lst;

  now = time( NULL );
  va_start( lst, format );
  printf( "[%08ld, %s:%s] \t", now, file, function );
  vprintf( format, lst );
  va_end( lst );
}

char* XklGetWindowTitle( Window w )
{
  Atom type_ret;
  int format_ret;
  unsigned long nitems, rest;
  unsigned char* prop;

  if ( Success == XGetWindowProperty( dpy, w, WM_NAME, 0L,
                                      -1L, False, XA_STRING, &type_ret,
                                      &format_ret, &nitems, &rest, &prop ) )
    return prop;
  else
    return NULL;
}

Bool XklGetAppState( Window appWin, XklState* state )
{
  Atom type_ret;
  int format_ret;
  unsigned long nitems, rest;
  unsigned char *prop;
  Bool ret = False;

  if ( ( XGetWindowProperty( dpy, appWin, XKLAVIER_STATE, 0L, PROP_LENGTH, False, XA_INTEGER,
                             &type_ret, &format_ret, &nitems, &rest,
                             &prop ) == Success ) &&
          ( type_ret == XA_INTEGER ) && ( format_ret == 8 ) )
  {
    if ( state != NULL )
    {
      state->group = prop[ 0 ];
      if ( state->group >= numGroups ||
              state->group < 0 )
        state->group = 0;
      XklDebug( "Obtained the state\n" );
    }
    if ( prop != NULL )
      XFree( prop );

    ret = True;
  }

  if ( ret )
    if ( state == NULL )
      XklDebug( "Appwin %ld, '%s' has some group id\n", appWin, XklGetDebugWindowTitle( appWin ) );
    else
      XklDebug( "Appwin %ld, '%s' has the group id %d\n", appWin, XklGetDebugWindowTitle( appWin ), state->group );
  else
    XklDebug( "Appwin %ld, '%s'  does not have group id\n", appWin, XklGetDebugWindowTitle( appWin ) );

  return ret;
}

void XklDelAppState( Window appWin )
{
  XDeleteProperty( dpy, appWin, XKLAVIER_STATE );
}

void XklSaveAppState( Window appWin, XklState * state )
{
  unsigned char prop[ 1 ];

  prop[ 0 ] = state->group;

  XChangeProperty( dpy, appWin, XKLAVIER_STATE, XA_INTEGER, 8, PropModeReplace,
                   prop, PROP_LENGTH );

  XklDebug( "Saved the group id %d for appwin %ld\n", state->group, appWin );
}

void XklSelectInput( Window win, long mask )
{
  if ( root == win )
    XklDebug( "** Someone is looking for %lX on root window ***\n", mask );

  XSelectInput( dpy, win, mask );
}

void XklLockGroup( int group, int restore )
{
  XklDebug( "## Posted request for change the group to %d ##\n", group );
  restoreState = restore;
  XkbLockGroup( dpy,
                XkbUseCoreKbd,
                group );
}

void XklSetLayoutPerApp( Bool isSet )
{
  layoutPerApp = isSet;
}

Bool XklIsLayoutPerApp( void )
{
  return layoutPerApp;
}
