#include "gswitchit.h"

#include <gdk/gdkx.h>
#include <gconf/gconf.h>

#include "../pixmaps/init.xpm"

#include "../xklavier/xklavier.h"

#include "switchcuts.h"

static GSwitchItApplet* theApplet;

static void GSwitchItAppletConfigChanged ( GConfClient *client,
                             guint cnxn_id,
                             GConfEntry *entry,
                             GSwitchItApplet* sia )
{
  gdk_beep();
  GSwitchItConfigUngrabSwitchcut( &sia->config );
  GSwitchItConfigGrabSwitchcut( &sia->config );
  GSwitchItAppletRevalidate( sia );

  XklSetLayoutPerApp( sia->config.layoutPerApp );
}

static void GSwitchItAppletStateCallback( int group, Bool restoreState, GSwitchItApplet *sia )
{
  XklDebug( "group is now %d restoreState: %d\n", group, restoreState );

  if ( restoreState == False )
  {
    if ( !GSwitchItConfigProcessSwitch( &sia->config, group ) )
    {
      XklDebug( "not going to process switch\n" );
      return;
    }

    if ( ( sia->config.secondaries & ( 1 << group ) ) != 0 &&
         !sia->config.allowSecondary )
    {
      XklDebug( "secondary -> go next\n" );
      GSwitchItConfigLockNextGroup( &sia->config );
      return; // we do not need to revalidate
    }
    else
    {
      if ( sia->config.forceNoBeep )
        sia->config.forceNoBeep = FALSE;
      else
        if ( sia->config.doBeep )
          gdk_beep();
    }
  }
  XklDebug( "do repaint\n" );
  GSwitchItAppletRevalidateGroup( sia, group );
  sia->config.allowSecondary = FALSE;
}

GdkFilterReturn GSwitchItAppletFilterXEvt( GdkXEvent *xevent,
                                           GdkEvent *event,
                                           GSwitchItApplet *sia)
{
  XKeyPressedEvent* kpevt;
  int evtype;
  int scid;
  guint sckey;

  XklFilterEvents( (XEvent*)xevent );

  evtype = ((XAnyEvent*)xevent)->type;

  if ( KeyPress == evtype )
  {
    kpevt = (XKeyPressedEvent*) xevent;

    XklDebug( "key pressed: %d\n", kpevt->keycode );
    scid = sia->config.switchcutId;
    if ( scid > 0 &&
         scid < total_switchcuts - 1 )
    {
      Switchcut * sc = switchcuts + scid;
      if ( sc->key )
      {
        sckey = XKeysymToKeycode( GDK_DISPLAY(), sc->key );
        if ( sckey && ( sckey == kpevt->keycode ) )
        {
          XklDebug( "configured switchcut detected\n" );
          GSwitchItConfigLockNextGroup( &sia->config );
        }
      }
    } else if ( scid == total_switchcuts - 1 )
    {
      if( ( sia->config.switchcutAutoKeycode == kpevt->keycode ) &&
          ( sia->config.switchcutState == ( kpevt->state & GROUP_MASK ) ) )
       {
          GSwitchItConfigLockNextGroup( &sia->config );
       }
    }
    XklDebug( "done\n" );
  }

  return GDK_FILTER_CONTINUE;
}

void GSwitchItAppletRevalidate( GSwitchItApplet* sia )
{
  XklState currentState;

  if ( XklGetState( XklGetCurrentWindow(), &currentState ) && 
       ( currentState.group >= 0 ) )
    GSwitchItAppletRevalidateGroup( sia, currentState.group );
}
	
void GSwitchItAppletRevalidateGroup( GSwitchItApplet* sia, int group )
{
  const char* pname;
  
  GdkPixbuf* image;
  XklDebug( "Revalidating for group %d\n", group );
    
  image = sia->config.images[group];
  if ( image != NULL )
  {
    GdkPixbuf       *scaled;
    PanelOrientType orient;
    int psize, xsize=0, ysize=0;
    double xyratio;

    if ( sia->config.activeImage )
    {
      gdk_pixmap_unref ( sia->config.activeImage );
    }
    if ( sia->config.activeMask )
    {
      gdk_bitmap_unref ( sia->config.activeMask );
    }

    orient = applet_widget_get_panel_orient( APPLET_WIDGET( sia->applet ) );
    psize = applet_widget_get_panel_pixel_size( APPLET_WIDGET( sia->applet ) ) - 4;
    xyratio = 1.0 * gdk_pixbuf_get_width( image )/
                    gdk_pixbuf_get_height( image );
    switch( orient )
    {
    case GNOME_Panel_ORIENT_UP:
    case GNOME_Panel_ORIENT_DOWN:
      ysize = psize;
      xsize = psize*xyratio;
      break;
    case GNOME_Panel_ORIENT_LEFT:
    case GNOME_Panel_ORIENT_RIGHT:
      xsize = psize;
      ysize = psize/xyratio;
      break;
    }

    scaled = gdk_pixbuf_scale_simple ( image,
                                       xsize,
                                       ysize,
                                       GDK_INTERP_HYPER );
    gdk_pixbuf_render_pixmap_and_mask( scaled,
                                       &sia->config.activeImage,
                                       &sia->config.activeMask, 128);
    gdk_pixbuf_unref( scaled );
    gtk_pixmap_set( GTK_PIXMAP( sia->drawingArea ),
                    sia->config.activeImage,
                    sia->config.activeMask );

    pname = XklGetGroupNames()[group];

    if ( pname )
      applet_widget_set_tooltip( APPLET_WIDGET( sia->applet ),
                                 pname );
  }
}

static void GSwitchItAppletRealize( GtkWidget *widget, GSwitchItApplet* sia )
{
  GSwitchItAppletRevalidate( sia );

  gdk_window_add_filter( NULL,
                         (GdkFilterFunc)GSwitchItAppletFilterXEvt,
                         sia );
  gdk_window_add_filter( GDK_ROOT_PARENT(),
                         (GdkFilterFunc)GSwitchItAppletFilterXEvt,
                         sia );

  XklSetLayoutPerApp( sia->config.layoutPerApp );

  XklStartListen();

  GSwitchItConfigGrabSwitchcut( &sia->config );

  GSwitchItAppletRevalidate( sia );
}

static void GSwitchItAppletUnrealize( GtkWidget *widget, GSwitchItApplet* sia )
{
  GSwitchItConfigUngrabSwitchcut( &sia->config );
}

static void GSwitchItAppletChangePixelSize( AppletWidget *widget, int size, GSwitchItApplet *sia )
{
  GSwitchItAppletRevalidate( sia );
}

static gint GSwitchItAppletButtonPressed ( GtkWidget	       *widget,
                                           GdkEventButton   *event,
                                           GSwitchItConfig  *sic )
{
  if ( event->button == 1 &&
       event->type == GDK_2BUTTON_PRESS )
      GSwitchItConfigLockNextGroup( sic );

  return TRUE;
}

static void GSwitchItAppletDestroyAbout( GtkWidget *w, GSwitchItApplet* sia )
{ sia->aboutBox=NULL; }

static void GSwitchItAppletCmdProps( AppletWidget *appwid, GSwitchItApplet* sia )
{
  int pid;

  pid = fork();
  if ( pid == -1 ) return;
  if ( pid == 0 )
  {
    execlp( "gswitchit-properties_capplet",
            "gswitchit-properties_capplet", NULL );
    exit( 127 );
  }
}

static void GSwitchItAppletCmdSetGroup( AppletWidget *appwid, gpointer data )
{
  XklState st;
  Window cur;

  theApplet->config.allowSecondary = TRUE;
  st.group = GPOINTER_TO_INT( data );

  cur = XklGetCurrentWindow();
  if ( cur != (Window)NULL )
  {
    XklSaveState( XklGetCurrentWindow(), &st );
    XSetInputFocus( GDK_DISPLAY(), cur, RevertToNone, CurrentTime );
  } else
  {
    // strange situation - bad can happen
    GSwitchItConfigApproveSwitch( &theApplet->config, st.group );
    XklLockGroup( st.group, False );
  }
}

static void GSwitchItAppletCmdHomePage( AppletWidget *appwid, GSwitchItApplet* sia )
{
  gnome_url_show ("http://gswitchit.sourceforge.net/");
}

static void GSwitchItAppletCmdAbout( AppletWidget *appwid, GSwitchItApplet* sia )
{
  GtkWidget* href;
  const gchar *authors[] = {"Sergey V. Udaltsov<svu@pop.convey.ru>", NULL};

  if( sia->aboutBox )
  {
    gtk_widget_show( sia->aboutBox );
    gdk_window_raise( sia->aboutBox->window );
    return;
  }
  sia->aboutBox = gnome_about_new( _(PACKAGE),
                                  VERSION,
                                  _("Copyright Sergey V. Udaltsov (C) 1999-2001"),
                                  authors,
                                  _("Keyboard switcher applet for the GNOME panel"),
                                  NULL );
  gtk_signal_connect( GTK_OBJECT( sia->aboutBox ),
                      "destroy",
                      GTK_SIGNAL_FUNC( GSwitchItAppletDestroyAbout),
                      sia );
  href = gnome_href_new( "http://gswitchit.sourceforge.net/",
                          _("GSwitchIt web page @ Sourceforge.net") );
  gtk_box_pack_start( GTK_BOX (GNOME_DIALOG (sia->aboutBox)->vbox),
                      href, FALSE, FALSE, 0 );

  gtk_widget_show( href );
  gtk_widget_show( sia->aboutBox );
}

static void GSwitchItAppletSetupMenu( GSwitchItApplet* sia )
{
  unsigned i, nGroups;
  const char ** pname;

  applet_widget_register_stock_callback( APPLET_WIDGET( sia->applet ),
                                         "about",
                                         GNOME_STOCK_MENU_ABOUT,
                                         _( "About..." ),
                                         GTK_SIGNAL_FUNC( GSwitchItAppletCmdAbout ),
                                         sia );

  applet_widget_register_stock_callback( APPLET_WIDGET( sia->applet ),
                                         "home",
                                         GNOME_STOCK_MENU_HOME,
                                         _( "Home page..." ),
                                         GTK_SIGNAL_FUNC( GSwitchItAppletCmdHomePage ),
                                         sia );

  applet_widget_register_stock_callback( APPLET_WIDGET( sia->applet ),
                                         "properties",
                                         GNOME_STOCK_MENU_PROP,
                                         _( "Properties" ),
                                         GTK_SIGNAL_FUNC( GSwitchItAppletCmdProps ),
                                         sia );

  applet_widget_register_stock_callback( APPLET_WIDGET( sia->applet ),
                                         "properties",
                                         GNOME_STOCK_MENU_PROP,
                                         _( "Properties" ),
                                         GTK_SIGNAL_FUNC( GSwitchItAppletCmdProps ),
                                         sia );

  applet_widget_register_callback_dir ( APPLET_WIDGET( sia->applet ),
                                        "Layouts",
                                        _( "Layouts" ) );

  nGroups = XklGetNumGroups();
  pname = XklGetGroupNames();
  for ( i = 0; i<nGroups; i++ )
  {
    char sz[40];
    g_snprintf( sz, sizeof(sz), "Layouts/%d", i );
    applet_widget_register_callback( APPLET_WIDGET( sia->applet ),
                                     sz,
                                     *pname++,
                                     GTK_SIGNAL_FUNC( GSwitchItAppletCmdSetGroup ),
                                     GINT_TO_POINTER( i ) );
  }
}

static void GSwitchItAppletInit( GSwitchItApplet* sia, int* pargc, char** argv )
{
  GdkPixbuf *initbuf;

  //!! Check the result - can be useful
  XklDebug( "GDK_DISPLAY(): %p, GDK_ROOT_WINDOW(): %ld\n",
    GDK_DISPLAY(), GDK_ROOT_WINDOW() );

  XklInit( GDK_DISPLAY() );

  //!! register callbacks here
  XklRegisterStateCallback( (StateCallback)GSwitchItAppletStateCallback, (void*)sia );

  GSwitchItConfigInit( &(sia->config), TRUE );

  gconf_client_notify_add( sia->config.confClient,
                           CONFIG_KEY_BEEP,
                           (GConfClientNotifyFunc)GSwitchItAppletConfigChanged,
                           sia,
                           NULL,
                           NULL );

  sia->applet = applet_widget_new( PACKAGE );

  if (!sia->applet)
      g_error("Can't create applet!\n");

  gtk_signal_connect( GTK_OBJECT( sia->applet ),
                      "realize",
                      GTK_SIGNAL_FUNC( GSwitchItAppletRealize ),
                      sia );

  gtk_signal_connect( GTK_OBJECT( sia->applet ),
                      "unrealize",
                      GTK_SIGNAL_FUNC( GSwitchItAppletUnrealize ),
                      sia );

  gtk_signal_connect( GTK_OBJECT ( sia->applet ),
                      "change_pixel_size",
                      GTK_SIGNAL_FUNC( GSwitchItAppletChangePixelSize ),
                      sia );

  sia->border = gtk_frame_new( NULL );
  gtk_frame_set_shadow_type ( GTK_FRAME( sia->border ), GTK_SHADOW_IN );

  initbuf = gdk_pixbuf_new_from_file ( gnome_pixmap_file( "no.xpm" ) );

  gdk_pixbuf_render_pixmap_and_mask( initbuf,
                                     &sia->config.activeImage,
                                     &sia->config.activeMask,
                                     128 );

  sia->drawingArea = gtk_pixmap_new( sia->config.activeImage,
                                     sia->config.activeMask );

  gdk_pixbuf_unref( initbuf );

  gtk_container_add( GTK_CONTAINER( sia->border ), sia->drawingArea );

  applet_widget_add( APPLET_WIDGET( sia->applet ), sia->border );

  applet_widget_set_tooltip( APPLET_WIDGET( sia->applet ), _(PACKAGE) );

  gtk_widget_realize( sia->applet );

  gtk_widget_add_events( sia->applet,
                         GDK_BUTTON_PRESS_MASK );

  gtk_signal_connect( GTK_OBJECT( sia->applet ),
                      "button_press_event",
                      GTK_SIGNAL_FUNC( GSwitchItAppletButtonPressed ),
                      sia );

  GSwitchItAppletSetupMenu( sia );
}

static void GSwitchItAppletTerm( GSwitchItApplet* sia )
{
  XklTerm();

  GSwitchItConfigTerm( &(sia->config), FALSE );
}

int main( int argc, char ** argv )
{
  GSwitchItApplet sia;
  GError *gconf_error = NULL;
  GConfEngine* eng;

  theApplet = &sia;

  bindtextdomain( PACKAGE, GNOMELOCALEDIR );
  textdomain( PACKAGE );

  applet_widget_init( PACKAGE, VERSION, argc, argv, NULL, 0, NULL );

  if (!gconf_init(argc, argv, &gconf_error))
  {
    g_warning( "Failed to init GConf: %s\n", gconf_error->message );
    g_error_free( gconf_error );
    return 1;
  }
  gconf_error = NULL;
  eng = gconf_engine_get_default();

  memset( &sia, 0, sizeof(sia) );

  GSwitchItAppletInit( &sia, &argc, argv );

  gtk_widget_show_all( GTK_WIDGET( sia.applet ) );

  applet_widget_gtk_main ();

  GSwitchItAppletTerm( &sia );
  gconf_engine_unref( eng );

  return 0;
}
